/*****************************************/
/* ATMEGA+MCP2515 SPI Interface Function */
/*****************************************/
/* Hardware : MCP2515 + AVRMEGA1280/2560 */
/*****************************************/

#include "mcp2515.h"

void mcp2515_init(uint8_t data_rate);
uint8_t spi_putc(uint8_t data);
void mcp2515_write_register(uint8_t data, uint8_t adress);
uint8_t mcp2515_read_register(uint8_t adress);
void mcp2515_bit_modify(uint8_t data, uint8_t mask, uint8_t adress);
void mcp2515_write_register_p(uint8_t adress, uint8_t *data, uint8_t length);

/*
 ****************************************
 * Initial MCP2515 Operation(Data Rate) *
 * -> CAN_62_5kbps                      *
 * -> CAN_125kbps                       *
 * -> CAN_250kbps                       *
 * -> CAN_500kbps                       *
 * -> CAN_1000kbps                      *
 ****************************************
 */
void mcp2515_init(uint8_t data_rate)
{
  ///* SPI Interface initialisieren */
  PORTB |=  (1<<SPI_CS);					//Disable CAN CS
  DDRB  |=  (1<<SPI_CS)|(1<<SPI_SCK)|(1<<SPI_MOSI);

  // Enable SPI = Master Mode  
  SPCR = (1<<SPE)|(1<<MSTR);				//Enable SPI, Master

  // SPI = High Speed
  // SPI2X:SPR1:SPR0 = 1:0:0(SCK=Fosc/2)
  // SPI2X:SPR1:SPR0 = 1:0:1(SCK=Fosc/8)
  // SPI2X:SPR1:SPR0 = 1:1:0(SCK=Fosc/32)
  // SPI2X:SPR1:SPR0 = 1:1:1(SCK=Fosc/64)
  // SCK = Fosc  / 2
  //     = 16MHz / 2
  //     = 8MHz
  SPSR |=  (1 << SPI2X);					//SPI2X = 1
  SPCR &= ~((1 << SPR1) | (1 << SPR0));		//SPR1:SPR0 = 0:0(Fosc/2) 

  // MCP2515 Software Reset 
  PORTB &= ~(1<<SPI_CS);					//Enable CS#(CS = Low)
  spi_putc(SPI_RESET);
  PORTB |= (1<<SPI_CS);						//Disable CS#(CS = High)

  /*********************************************************************************
   *	CAN Bit Timings With Oscillator 20MHz
   *	Fosc 		  	= 20 MHz 
   *    Tosc 		  	= 1 / 20MHz = 50nS
   *    1TQ Minimum     = 2 Tosc 
   *
   *    1 Bit Time CAN = Sync Seg + Prop Seg + Phase Seg1(PS1) + Phase Seg2(PS2)
   *	Sync Seg   	= Fixed = 1TQ                       Best Quality = 1 TQ
   *	Prop Seg   	= (PRSEG0[0..7] + 1) * TQ = 1..8 TQ Best Quality = 3 TQ+
   *	Phase Seg1 	= (PHSEG1[0..7] + 1) * TQ = 1..8 TQ Best Quality = 3 TQ+
   *	Phase Seg2 	= (PHSEG2[0..7] + 1) * TQ = 1..8 TQ Best Quality = 3 TQ+
   *
   *    1TQ For 1 Bit Time  = [4..25]TQ Best Quality = 10 TQ(1+3+3+3)
   *    -> SJW:    Synchronization Jump Width Length = 1([0:0]+1)
   *    -> PRSEG:  Propagation Segment Length = 3([0:1:0]+1)
   *    -> PHSEG1: PS1 Length = 3([0:1:0]+1)
   *    -> PHSEG2: PS2 Length = 3([0:1:0]+1)
   *
   *    -> 62.5Kbps     = 1 / 62.5KHz = 16uS   
   *       Bit Time     = 16uS 
   *       Want 10TQ    = 16uS
   *       1TQ          = 16uS / 10 = 1.6uS
   *       1TQ:Tosc     = 1.6uS / 50ns
   *                    = 32 Tosc = 1600nS
   *       BRP          = (32 / 2)-1
   *                    = 15(0:0:1:1:1:1)   
   *
   *    -> 125Kbps      = 1 / 125KHz  = 8uS   
   *       Bit Time     = 8uS  
   *       Want 10TQ    = 8uS
   *       1TQ          = 8uS / 10 = 0.8uS
   *       1TQ:Tosc     = 0.8uS / 50ns
   *                    = 16 Tosc = 800nS
   *       BRP          = (16 / 2)-1
   *                    = 7(0:0:0:1:1:1)   
   *
   *    -> 250Kbps    	= 1 / 250KHz  = 4uS
   *       Bit Time  	= 4uS  
   *       Want 10TQ    = 4uS
   *       1TQ          = 4uS / 10 = 0.4uS
   *       1TQ:Tosc     = 0.4uS / 50ns
   *                    = 8 Tosc = 400nS
   *       BRP          = (8 / 2)-1
   *                    = 3(0:0:0:0:1:1)   
   *
   *    -> 500Kbps  	= 1 / 500KHz  = 2uS
   *       Bit Time 	= 2uS  
   *       Want 10TQ    = 2uS
   *       1TQ          = 2uS / 10=0.2uS
   *       1TQ:Tosc     = 0.2uS / 50ns
   *                    = 4 Tosc = 200nS
   *       BRP          = (4 / 2)-1
   *                    = 1(0:0:0:0:0:1)   
   *
   *    -> 1Mbps        = 1 / 1MHz = 1uS   
   *       Bit Time  	= 1uS  
   *       Want 10TQ    = 1uS
   *       1TQ          = 1uS / 10=0.1uS
   *       1TQ:Tosc     = 0.1uS / 50ns
   *                    = 2 Tosc = 100nS
   *       BRP          = (2 / 2)-1
   *                    = 0(0:0:0:0:0:0)   
   *********************************************************************************/

  /*********************************************************************************
   *	CAN Bit Timings With Oscillator 16MHz
   *	Fosc 		  	= 16 MHz 
   *    Tosc 		  	= 1 / 16MHz = 62.5nS
   *    1TQ Minimum     = 2 Tosc 
   *
   *    1 Bit Time CAN = Sync Seg + Prop Seg + Phase Seg1(PS1) + Phase Seg2(PS2)
   *	Sync Seg   	= Fixed = 1TQ                       Best Quality = 1 TQ
   *	Prop Seg   	= (PRSEG0[0..7] + 1) * TQ = 1..8 TQ Best Quality = 3 TQ+
   *	Phase Seg1 	= (PHSEG1[0..7] + 1) * TQ = 1..8 TQ Best Quality = 2 TQ+
   *	Phase Seg2 	= (PHSEG2[0..7] + 1) * TQ = 1..8 TQ Best Quality = 2 TQ+
   *
   *    1TQ For 1 Bit Time  = [4..25]TQ Best Quality = 8 TQ(1+3+2+2)
   *    -> SJW:    Synchronization Jump Width Length = 1([0:0]+1)
   *    -> PRSEG:  Propagation Segment Length = 3([0:1:0]+1)
   *    -> PHSEG1: PS1 Length = 2([0:0:1]+1)
   *    -> PHSEG2: PS2 Length = 2([0:0:1]+1)
   *
   *    -> 62.5Kbps     = 1 / 62.5KHz = 16uS   
   *       Bit Time     = 16uS 
   *       Want 8TQ     = 16uS
   *       1TQ          = 16uS / 8 = 2uS
   *       1TQ:Tosc     = 2uS / 62.5ns
   *                    = 32 Tosc = 2000nS
   *       BRP          = (32 / 2)-1
   *                    = 15(0:0:1:1:1:1)   
   *
   *    -> 125Kbps      = 1 / 125KHz  = 8uS   
   *       Bit Time     = 8uS  
   *       Want 8TQ     = 8uS
   *       1TQ          = 8uS / 8 = 1uS
   *       1TQ:Tosc     = 1uS / 62.5ns
   *                    = 16 Tosc = 1000nS
   *       BRP          = (16 / 2)-1
   *                    = 7(0:0:0:1:1:1)   
   *
   *    -> 250Kbps    	= 1 / 250KHz  = 4uS
   *       Bit Time  	= 4uS  
   *       Want 8TQ     = 4uS
   *       1TQ          = 4uS / 8 = 0.5uS
   *       1TQ:Tosc     = 0.5uS / 62.5ns
   *                    = 8 Tosc = 400nS
   *       BRP          = (8 / 2)-1
   *                    = 3(0:0:0:0:1:1)   
   *
   *    -> 500Kbps  	= 1 / 500KHz  = 2uS
   *       Bit Time 	= 2uS  
   *       Want 8TQ     = 2uS
   *       1TQ          = 2uS / 8 = 0.25uS
   *       1TQ:Tosc     = 0.25uS / 62.5ns
   *                    = 4 Tosc = 200nS
   *       BRP          = (4 / 2)-1
   *                    = 1(0:0:0:0:0:1)   
   *
   *    -> 1Mbps        = 1 / 1MHz    = 1uS   
   *       Bit Time  	= 1uS  
   *       Want 8TQ     = 1uS
   *       1TQ          = 1uS / 8 = 0.125uS
   *       1TQ:Tosc     = 0.125uS / 62.5ns
   *                    = 2 Tosc = 100nS
   *       BRP          = (2 / 2)-1
   *                    = 0(0:0:0:0:0:0)   
   *********************************************************************************/

  
  /* Start of CAN Baudrate Generate Config */	

  //REQOP = 1:0:0(Set Configuration mode)
  mcp2515_bit_modify(CANCTRL,0xE0,(1<<REQOP2));
  
  //Configure Baud Rate Prescaler
  switch(data_rate)     // interpret command
  {
    				   //Set to 62.5 kbps
                       //SJW=00(Length=1 TQ), BRP = 15(TQ = 2 x (15+1)/FOSC)
                       //CNF1 = 0:0 + 0:0:1:1:1:1 -> 62.5 Kbps
    case CAN_62_5kbps: mcp2515_write_register(CNF1,(1<<BRP3)|(1<<BRP2)|(1<<BRP1)|(1<<BRP0)); 
                       break;

					   //Set to 125 kbps
                       //SJW=00(Length=1 TQ), BRP = 7(TQ = 2 x (7+1)/FOSC)
                       //CNF1 = 0:0 + 0:0:0:1:1:1 -> 125  Kbps
    case CAN_125kbps:  mcp2515_write_register(CNF1,(1<<BRP2)|(1<<BRP1)|(1<<BRP0));        	                   
                       break;

                       //Set to 250 kbps
                       //SJW=00(Length=1 TQ), BRP = 3(TQ = 2 x (3+1)/FOSC)
					   //CNF1 = 0:0 + 0:0:0:0:1:1 -> 250  Kbps
    case CAN_250kbps:  mcp2515_write_register(CNF1,(1<<BRP1)|(1<<BRP0));         
                       break;

                       //Set to 500 kbps
					   //SJW=00(Length=1 TQ), BRP = 1(TQ = 2 x (1+1)/FOSC) 
                       //CNF1 = 0:0 + 0:0:0:0:0:1 -> 500  Kbps
    case CAN_500kbps:  mcp2515_write_register(CNF1,(1<<BRP0));        
                       break;

                       //Set to 1000 kbps
                       //SJW=00(Length=1 TQ), BRP = 0(TQ = 2 x (0+1)/FOSC)
                       //CNF1 = 0:0 + 0:0:0:0:0:0 -> 1000 Kbps
    case CAN_1000kbps: mcp2515_write_register(CNF1,0);    		
                       break;

                       //unrecognized or null command
    default:           break;         
  }


#if(F_CAN == 20000000UL)
  //If OSC = 20MHz 
  //BTLMODE=1,PHSEG1=3TQ(0:1:0),PRSEG=3TQ(0:1:0)
  //CNF2 = 1+0+0:1:0+0:1:0 
  mcp2515_write_register(CNF2,(1<<BTLMODE)|(1<<PHSEG11)|(1<<PRSEG1));
  
  //PHSEG2=3TQ(0:1:0)
  //CNF3 = 0+0+x:x:x:0:1:0 
  mcp2515_write_register(CNF3,(1<<PHSEG21));
#endif
  
#if(F_CAN == 16000000UL)  
  //If OSC = 16MHz 
  //BTLMODE=1,PHSEG1=3TQ(0:1:0),PRSEG=2TQ(0:0:1)
  //CNF2 = 1+0+0:1:0+0:0:1 
  mcp2515_write_register(CNF2,(1<<BTLMODE)|(1<<PHSEG11)|(1<<PRSEG0));

  //PHSEG2=2TQ(0:0:1)
  //CNF3 = 0+0+x:x:x:0:0:1 
  mcp2515_write_register(CNF3,(1<<PHSEG20));
#endif
  /* End of CAN Baudrate Generate Config */

 	
  //Enable RX1,RX0 all Interrupts 
  mcp2515_write_register(CANINTE,(1<<RX1IE)|(1<<RX0IE)); 
		
  // Buffer 0 : Receive All Message
  mcp2515_write_register(RXB0CTRL,(1<<RXM1)|(1<<RXM0));
	
  //Buffer 1 : Receive All Message
  mcp2515_write_register(RXB1CTRL,(1<<RXM1)|(1<<RXM0));

  //Clear All Data Register	
  uint8_t temp[4] = { 0, 0, 0, 0 };
	
  // Filter Buffer 0 
  mcp2515_write_register_p(RXF0SIDH, temp, 4);
  mcp2515_write_register_p(RXF1SIDH, temp, 4);
	
  // Filter Buffer 1 
  mcp2515_write_register_p(RXF2SIDH, temp, 4);
  mcp2515_write_register_p(RXF3SIDH, temp, 4);
  mcp2515_write_register_p(RXF4SIDH, temp, 4);
  mcp2515_write_register_p(RXF5SIDH, temp, 4);
	
  // Maske Buffer 0 
  mcp2515_write_register_p(RXM0SIDH, temp, 4);
	
  // Maske Buffer 1 
  mcp2515_write_register_p(RXM1SIDH, temp, 4);
	
  
  //Enable RX1BF,RX0BF Output Pins
  mcp2515_write_register(BFPCTRL,(1<<B1BFE)|(1<<B0BFE)|(1<<B1BFM)|(1<<B0BFM));
	
  //Disable TX0RTS,TX1RTS,TX2RTS Input Pin
  mcp2515_write_register( TXRTSCTRL, 0);
	
  //REQOP = 0:0:0 (Set Normal Operation mode)
  mcp2515_bit_modify(CANCTRL, 0xE0, 0);

  //REQOP = 0:1:0 (Set Loopback Operation mode)
  //mcp2515_bit_modify(CANCTRL, 0xE0, (1<<REQOP1));
}

/*
 *************************
 * Send/Receive SPI Byte *
 *************************
 */
uint8_t spi_putc(uint8_t data)
{
  SPDR = data;						//Send Byte SPI Data
  while(!(SPSR & (1<<SPIF))); 		//Wait Transmission Complete
	
  return SPDR;						//Receive Data
}

/*
 *******************************
 * Write MCP2515 Register Data *
 ******************************* 
 */
void mcp2515_write_register(uint8_t adress, uint8_t data)
{
  PORTB &= ~(1<<SPI_CS);			//Enable CS#(CS = Low)
	
  spi_putc(SPI_WRITE);				//Instruction	
  spi_putc(adress);					//Address	
  spi_putc(data);					//Data
	
  PORTB |= (1<<SPI_CS);				//Disable CS#(CS = High)
}

/*
 ******************************
 * Read MCP2515 Register Data *
 ******************************
 */
uint8_t mcp2515_read_register(uint8_t adress)
{
  uint8_t data;
	
  PORTB &= ~(1<<SPI_CS);			//Enable CS#(CS = Low)
	
  spi_putc(SPI_READ);				//Instruction
  spi_putc(adress);					//Address
  data = spi_putc(0xFF);			//Dummy Read Data	
	
  PORTB |= (1<<SPI_CS);				//Disable CS#(CS = High)
	
  return data;
}

/*
 **************************
 * Read MCP2515 RX Buffer *
 **************************
 */
uint8_t mcp2515_read_rx_buffer(uint8_t adress)
{
  uint8_t data;
	
  //Address Out of Length
  if (adress & 0xF9)
  {
  	return 0;
  }
  	  
  PORTB &= ~(1<<SPI_CS);			//Enable CS#(CS = Low)
	
  spi_putc(SPI_READ_RX | adress);	//Instruction	
  data = spi_putc(0xFF);			//Dummy Read Data
	
  PORTB |= (1<<SPI_CS);				//Disable CS#(CS = High)
	
  return data;
}

/* 
 *************************************
 * Write MCP2515 Bit Modify Register *
 *************************************
 */
void mcp2515_bit_modify(uint8_t adress, uint8_t mask, uint8_t data)
{
  PORTB &= ~(1<<SPI_CS);			//Enable CS#(CS = Low)
	
  spi_putc(SPI_BIT_MODIFY);			//Instruction	
  spi_putc(adress);					//Address	
  spi_putc(mask);					//Mask	
  spi_putc(data);					//Data
	
  PORTB |= (1<<SPI_CS);				//Disable CS#(CS = High)
}

/*
 *************************************
 * Write MCP2515 Register Multi-Byte *
 *************************************
 */
void mcp2515_write_register_p(uint8_t adress, uint8_t *data, uint8_t length)
{
  uint8_t i;
 	
  PORTB &= ~(1<<SPI_CS);			//Enable CS#(CS = Low)
	
  spi_putc(SPI_WRITE);				//Instruction	
  spi_putc(adress);					//Address	
  for (i=0; i<length ;i++)			//Write Data = Length
  {
    spi_putc(*data++);
  }
  	
  PORTB |= (1<<SPI_CS);				//Disable CS#(CS = High)
}

/*
 ************************************
 * Read MCP2515 Register Multi-Byte *
 ************************************
 */
void mcp2515_read_register_p( uint8_t adress, uint8_t *data, uint8_t length )
{
  uint8_t i;
	
  PORTB &= ~(1<<SPI_CS);			//Enable CS#(CS = Low)
	
  spi_putc(SPI_READ);				//Instruction	
  spi_putc(adress);					//Address	
  for (i=0; i<length ;i++ )			//Read Data = Length
  {
  	*data++ = spi_putc(0xFF);
  }
  	
  PORTB |= (1<<SPI_CS);				//Disable CS#(CS = High)
}

